//******************************************************************************
//   MSP-FET430P140 Demo - SMBus Slave Transmits to MSP430 Master
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master 
//   reads from the slave. This is the slave code. The master code is called
//   fet140_SMB_mstr_slvrst.c. The TX data begins at 1 and is incremented 
//   each time it is sent.
//   The TXRDYIFG interrupt is used to know when to TX.
//   TimerB0 is used as capture input for SCL transitions. TimerA1 is used to 
//   detect Timeout. TimerA2 is used to simulate the delay between data bytes.
//   P1.0(LED) is toggled on a timeout
//   ACLK = n/a, MCLK = SMCLK = I2CIN = DCO ~ 800kHz
//   //* MSP430F169 Device Required *//
//   
//                                 /|\  /|\
//                  MSP430F169     10k  10k     MSP430F169
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1             |
//             |                 |  |      |             P1.0|-->LED
//             |                 |  |      |                 |
//    P4.0(TB0)|<----------- P3.3|<-+----->|P3.3             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  Feb 2005
//  Built with IAR Embedded Workbench Version: 3.21A
//******************************************************************************

#include  <msp430x16x.h>

typedef enum {
             SMBS_MISC               = (unsigned char) 0x48,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/	 

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);            

char TXData = 1;

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P3SEL |= 0x0A;                            // Select I2C pins
  P1DIR |= 0xFF;                            // P1.0 output
                                            // unused ports to output
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFE;
  P5DIR =  0xFF;
  P6DIR =  0xFF;
  P1OUT &= ~0x01;                           // Clear P1.0 
  TACTL = TASSEL_2;                         // SMCLK
  CCTL1 = CCIE;
  CCR0 = 30000;                         
  CCR1 = 20000;                             // Detect Timeout
  CCR2 = 22000;                             // Generate delay between data bytes
  P4SEL |= 0x01;                            // Select TB0 Input for SCL
  TBCTL = TBSSEL_2;                         // SMCLK
  TBCCTL0 = CAP + SCS + CCIE;               // Sync Capture
  P6OUT = 0;
  TBCCTL0 |= CM_2;                          // Falling Edge 
  SMBS_Init (SMBS_MISC,0x48);               // Own address configured as 48h

  _BIS_SR(CPUOFF+GIE);                      // Enter LPM0, Enable interrupts
   while(1)
   {
     TACCTL2 = CCIE;    
     TACTL |= TACLR + MC_1;                 // Up Mode
     LPM0;
     TACCTL2 &= ~CCIE;
     TACTL &= ~MC_1;
     I2CIE = TXRDYIE;     
     
   }
}

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  U0CTL |= I2C + SYNC;                      // Recommended init procedure
  U0CTL &= ~I2CEN;                          // Recommended init procedure
  I2CTCTL |= I2CSSEL1;                      // SMCLK
  if (Add_Param == SMBS_MISC)
  {
      I2COA = Slave_Add;                    // Own Address is 048h
  } 
  else
      I2COA = (unsigned char) Add_Param ;
  
  I2CIE = TXRDYIE;                          // Enable TXRDYIFG interrupt
  U0CTL |= I2CEN;                           // Enable I2C
}

// Common ISR for I2C Module
#pragma vector=USART0TX_VECTOR
__interrupt void I2C_ISR(void)
{
 
 switch( I2CIV )
 {
   case  2: break;                          // Arbitration lost
   case  4: break;                          // No Acknowledge
   case  6: break;                          // Own Address
   case  8: break;                          // Register Access Ready
   case 10: break;                          // Receive Ready
   case 12:                                 // Transmit Ready
     I2CIE &= ~TXRDYIE;
     I2CDRB = TXData++;                     // Load I2CDRB and increment
     _BIC_SR_IRQ(CPUOFF);                   // Clear LPM0
     break;
   case 14: break;                          // General Call
   case 16: break;                          // Start Condition
 }
}

// Timer B0 interrupt service routine
#pragma vector=TIMERB0_VECTOR
__interrupt void tb0_isr(void)
{
     TACTL &= ~MC_1;                        // Stop TimerA0
     TACTL |= TACLR;
     TACTL |= MC_1;                         // Start TimerA0 in up mode
}

// Timer A1,A2 interrupt service routine
// Timeout has happened or Next Byte in Time
#pragma vector=TIMERA1_VECTOR
__interrupt void Timer_A (void)
{
  switch( TAIV )
 {
   case  2: 
            TACTL &= ~MC_1;                 // Recomended Timer Halt procedure
            TACTL |= TACLR;
            P1OUT ^= 0x01;
            SMBS_Init (SMBS_MISC,0x48);     // Slave address configured as 48h
            _BIC_SR_IRQ(CPUOFF);
            break;                          // TACCR1
   case  4: 
            _BIC_SR_IRQ(CPUOFF);            // Clear LPM0
            break;                          // TACCR2
   case  6: break;                          // 
   case  8: break;                          // 
   case 10: break;                          // TAIFG
   case 12: break;
   case 14: break;                          // 
   case 16: break;                          // 
 }
  
   
}
